﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net.Http.Headers;
using System.Net.Http;
using System.Text;
using System.Threading.Tasks;
using System.Security.Cryptography;
using System.Diagnostics;
using System.Net;
using System.Threading;

namespace FileUpload
{
    internal class Program
    {
        private static readonly SemaphoreSlim semaphoreSlim = new(1, 1);

        private const int CHUNK_SIZE = 16 * 1024 * 1024; // 16MB
        static async Task Main(string[] args)
        {
            //Replace with your own implementation of ZiskejTokenProNahravaniSouboru
            var (token, odkaz) = await IENCommunicator.ZiskejTokenProNahravaniSouboru();
            Console.WriteLine($"Token: {token}");

            //Add your desired files
            var files = Directory.GetFiles(@".\files");

            using var sha512 = SHA512.Create();
            using var handler = new HttpClientHandler
            {
                UseCookies = false
            };
            var tasks = files.Select(async filePath =>
            {
                using var httpClient = new HttpClient(handler, false) { Timeout = TimeSpan.FromMinutes(15) };
                var fileInfo = new FileInfo(filePath);
                var totalLength = fileInfo.Length;
                var fileName = fileInfo.Name;
                using var fileStream = File.OpenRead(fileInfo.FullName);

                Console.WriteLine($"Uploading {fileInfo.Name}");
                var etag = await GetEtagAsync(httpClient, odkaz, fileName, totalLength);
                if (string.IsNullOrWhiteSpace(etag))
                {
                    return;
                }
                Console.WriteLine($"\tETag: {etag}.");
                if (await UploadFileInChunksAsync(httpClient, odkaz, fileName, fileStream, etag))
                {
                    //optional hash to check after upload
                    fileStream.Position = 0;
                    var digest = $"sha-512={Convert.ToBase64String(sha512.ComputeHash(fileStream))}";
                    var docId = await FinalizeFileUploadAsync(httpClient, odkaz, fileName, etag, digest);
                    Console.WriteLine($"\tDocId: {Uri.UnescapeDataString(docId ?? "")}");
                }
            });
            await Task.WhenAll(tasks);
            Console.Write($"Press any key to continue...");
            Console.ReadKey();
        }

        static async Task<string> GetEtagAsync(HttpClient httpClient,
                                               string uploadUrl,
                                               string fileName,
                                               long totalLength)
        {
            var tagContent = new StringContent("");
            tagContent.Headers.Add("File-Name", Uri.EscapeDataString(fileName));
            tagContent.Headers.Add("File-Size", totalLength.ToString());

            var response = await httpClient.PostAsync(
                $"{uploadUrl}/create-etag",
                tagContent);

            if (!response.IsSuccessStatusCode)
            {
                Console.WriteLine($"\tEtag creation failed: {response.StatusCode} {await response.Content.ReadAsStringAsync()}");
                return null;
            }

            var etag = await response.Content.ReadAsStringAsync();
            return etag;
        }

        static async Task<bool> UploadFileInChunksAsync(HttpClient httpClient,
                                                        string uploadUrl,
                                                        string fileName,
                                                        FileStream fileStream,
                                                        string etag)
        {
            var buffer = new byte[CHUNK_SIZE];
            int bytesRead;

            var startPosition = 0;
            while ((bytesRead = await fileStream.ReadAsync(buffer, 0, CHUNK_SIZE)) > 0)
            {
                var endPosition = startPosition + bytesRead - 1;
                var uploadContent = new ByteArrayContent(buffer, 0, bytesRead);
                uploadContent.Headers.ContentType = new MediaTypeHeaderValue("application/octet-stream");
                uploadContent.Headers.Add("File-Name", Uri.EscapeDataString(fileName));
                uploadContent.Headers.Add("File-Etag", etag);
                uploadContent.Headers.Add("Content-Range", $"bytes {startPosition}-{endPosition}/*");

                var response = await httpClient.PostAsync(
                    $"{uploadUrl}/upload",
                    uploadContent);

                if (!response.IsSuccessStatusCode)
                {
                    Console.WriteLine($"\tUpload failed for {etag}: {response.StatusCode} {await response.Content.ReadAsStringAsync()}");
                    return false;
                }

                Console.WriteLine("\tChunk uploaded successfully.");

                startPosition = endPosition + 1;
            }
            return true;
        }

        static async Task<string> FinalizeFileUploadAsync(HttpClient httpClient,
                                                          string uploadUrl,
                                                          string fileName,
                                                          string etag,
                                                          string sha512 = null)
        {
            var finalizeContent = new StringContent("");
            finalizeContent.Headers.Add("File-Name", Uri.EscapeDataString(fileName));
            finalizeContent.Headers.Add("File-Etag", etag);
            if (!string.IsNullOrEmpty(sha512))
            {
                finalizeContent.Headers.Add("Content-Digest", sha512);
            }
            var response = await httpClient.PostAsync(
                    $"{uploadUrl}/upload?finalized=true",
                    finalizeContent);

            if (!response.IsSuccessStatusCode)
            {
                Console.WriteLine($"\tFinalization failed: {response.StatusCode} {await response.Content.ReadAsStringAsync()}");
                return null;
            }

            return await response.Content.ReadAsStringAsync();
        }
    }
}
